//============================================================================
// Name        : gfuc.cpp
// Author      : Jiří Čejka (jirkacejka AT gmail DOT com)
// Version     : 0.1
// Date        : 2014/02/24
// Copyright   : GPLv3
// Description : This program converts gEDA PCB fp file (footprint) from imperial
//               to metric units and vice versa
// Usage       : ./gfuc --help
//               ./gfuc --in <filename> --out <filename> --[mil2mm|mm2mil]
//============================================================================

#include <string>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <cmath>
#include <sys/stat.h>
#include <vector>
#include <sstream>
#include <iterator>
#include <cctype>


// Amount of inches in 1 meter - constant used for unit conversion.
const float conversionConst = 39.3700787402;


/**
 * Prints help message if something goes wrong or user wants to see this message
 */
void printHelpMessage() {
	std::cout << "gEDA footprint unit convertor\n" << std::endl;
	std::cout << "Usage is --in <infile> --out <outfile> --[mm2mil|mil2mm]\n"; // Inform the user of how to use the program
}


/**
 * Converts value from mm to mil or mil to mm then round it the result to:
 * 1) 4 decimal places for mm as converted value or
 * 2) 2 decimal places for mil as converted value
 *
 * @param value
 * @param direction - string representing direction of unit conversion as typed to command line
 * @return converted value as float
 */
float convert(float value, std::string direction) {
	float convertedValue = 0;

	if (direction == "--mil2mm")
		convertedValue = round(value / conversionConst * 10000) / 10000;
	else if (direction == "--mm2mil")
		convertedValue = round(value * conversionConst * 100) / 100;

	return convertedValue;
}


/**
 * Converts float to its string representation
 *
 * @param value - value to be converted into it's string representation
 * @param n - number of decimal precision
 * @return precised value as a string
 */
template <typename T>
std::string toPrecisedString(const T value, const int n = 4) {
    std::ostringstream out;
    out << std::fixed << std::setprecision(n) << value;
    return out.str();
}


/**
 * Take line param, split it, converts numbers and compose new line with converted numbers
 *
 * @param line - string represents line readed from source file
 * @return line with converted values as a string
 */
std::string convertLine(std::string line, std::string direction) {

	std::string retVal = "";

	// Split line by 'space' into vector<string>.
	std::istringstream buf(line);
	std::istream_iterator<std::string> beg(buf), end;

	std::vector<std::string> tokens(beg, end); // done!

	// Iterate thru vector and converts values as desired.
	std::vector<std::string>::iterator it;

	for (it = tokens.begin(); it != tokens.end(); ++it) {
		std::string new_s = "";
		/// Find first position of number in token (digit or '-' character)
		/// and all character found before the number copy to new string
		/// which is going to act as replacement.
		unsigned int startpos = 0;
		for(unsigned int i = 0; i<(*it).size(); i++) {
			char c = (*it)[i];
			if (isdigit(c) || c == '-') {
				startpos = i;
				break;
			} else new_s += c;
		}
		std::string s = (*it).substr(startpos,std::string::npos);

		if(direction == "--mil2mm") {
			size_t pos = s.find("mil",0);
			if(pos != std::string::npos) {
				// Get float from string.
				std::istringstream stm(s) ;
				float f;
				stm >> f;
				// Convert float to other units.
				float nf = convert(f,direction);

				// Build new string with converted value and rest of the formal string.
				bool copied = false;
				for(char& c : s) {
					if (isdigit(c) || c == '-' || c == '.' || c == 'm' || c == 'i' || c == 'l') {
						if (!copied) {
							// Replace formal value with converted and include units (format is: #.0000mm).
							new_s += toPrecisedString(nf, 4);
							new_s += "mm";
							copied = true;
						} else continue;
					} else new_s += c;
				}

				// Replace appropriate value in vector<string> with new one including converted value.
				tokens.erase(it);
				tokens.insert(it, new_s);
			} else {
// debug	        		cout << " << This tocken will not be parsed \n";
			}
		} else if (direction == "--mm2mil") {
			size_t pos = s.find("mm",0);
			if(pos != std::string::npos) {
				// Get float from string.
				std::istringstream stm(s) ;
				float f;
				stm >> f;
				// Convert float to other units.
				float nf = convert(f,direction);

				// Build new string with converted value and rest of the formal string.
				bool copied = false;
				for(char& c : s) {
					if (isdigit(c) || c == '-' || c == '.' || c == 'm') {
						if (!copied) {
							// Replace formal value with converted and include units (format is: #.0000mm).
							new_s += toPrecisedString(nf, 2);
							new_s += "mil";
							copied = true;
						} else continue;
					} else new_s += c;
				}

				// Replace appropriate value in vector<string> with new one including converted value.
				tokens.erase(it);
				tokens.insert(it, new_s);
			} else {
// debug	        		cout << " << This token will not be parsed \n";
			}
		}
		// TODO Handle the direction inproper value here with some message to user and exit the program (throw some exception?).
	}

	// Compose of line with converted values.
	for(it = tokens.begin(); it != tokens.end(); ++it) {
		retVal += *it;
		retVal += ' ';
	}

// debug		cout << retVal << endl;
	return retVal;
}


/**
 * Test if file by given name already exists
 *
 * @param name - filename to e tested for existence
 * @return true if file specified by its name exists
 *         false otherwise
 */
inline bool fileExists(const std::string& name) {
	struct stat buffer;
	return (stat (name.c_str(), &buffer) == 0);
}



/**
 * MAIN
 */
int main(int argc, char* argv[]) {

	/// Parsing command line arguments and check the value of argc.
	/// If not enough parameters have been passed, inform user and exit.
	/// TODO implement --oldformat parameter which will cause conversion
	///      of old format footprint
    if ((argc == 6) &&
       (std::string(argv[1]) != "--help") &&
       (std::string(argv[1]) == "--in") &&
       (std::string(argv[3]) == "--out") &&
       ((std::string(argv[5]) == "--mil2mm") || (std::string(argv[5]) == "--mm2mil"))) {
		// Check existence of input and output files.
		if(fileExists(argv[2])) { // test input file
			if(fileExists(argv[4])) { // test output file
				std::cout << "Output file exists, what I have to do? Exit, rename or rewrite file? [E,e|R,r|W,w]: ";
				//TODO what we are going to do. Exit program or rewrite output file? Implement the desired behaviour.
				return 0;
			} else { // Output file does not exist, we are going to create it later in this branch.
				std::ifstream inputFootprintFile; // input file
				std::ofstream outputFootprintFile; // output file
				inputFootprintFile.open(argv[2]);
				outputFootprintFile.open (argv[4]);

				/// Open source file, read it line by line and convert as desired.
				/// Result then store into the output file
				if (inputFootprintFile.is_open()) {
					std::string sourceline, destline;
					while (!inputFootprintFile.eof() && getline (inputFootprintFile,sourceline)) {
// debug						cout << sourceline << '\n';
						destline = convertLine(sourceline, std::string(argv[5]));
						outputFootprintFile << destline << '\n';
					}
				}

				inputFootprintFile.close();
				outputFootprintFile.close();

				std::cout << "Conversion done successfully!" << std::endl;

				return 0;
			}
		} else {
			std::cout << "ERR: Input filename invalid, file does not exist.";
		}
    }

    printHelpMessage();
	return 0;
}
